<?php

namespace App\Http\Livewire\Employees;

use App\Traits\HasBulkActions;
use App\Traits\HasCachedRows;
use App\Traits\HasPerPagePagination;
use App\Traits\HasSort;
use Employees\Models\Employee;
use Illuminate\Support\Arr;
use Livewire\Component;

class EmployeesTable extends Component
{
    use HasPerPagePagination, HasSort, HasBulkActions, HasCachedRows;

    public $queryString = [];
    public $showEditModal = false;
    public $showFilters = false;
    public Employee $editing;
    public $showDeleteModal = false;
    public $filters = [
      'search' => '',
      'active' => ''
    ];


    public function rules()
    {
        return [
            'editing.name' => 'required',
            'editing.surname' => 'required',
            'editing.active' => 'nullable',
            'editing.contact_information.address' => 'required',
            'editing.contact_information.number' => 'required|numeric',
            'editing.contact_information.addon' => 'nullable',
            'editing.contact_information.zip_code' => 'required',
            'editing.contact_information.city' => 'required',
            'editing.contact_information.country' => 'required',
        ];
    }

    public function messages()
    {
        return [
            'editing.name.required' => __('Naam is verplicht'),
            'editing.surname.required' => __('Achternaam is verplicht'),
            'editing.contact_information.address.required' => __('Adres is verplicht'),
            'editing.contact_information.number.required' => __('Nummer is verplicht'),
            'editing.contact_information.number.numeric' => __('Nummer moet uit cijfers bestaan'),
            'editing.contact_information.zip_code.required' => __('Postcode is verplicht'),
            'editing.contact_information.city.required' => __('Woonplaats is verplicht'),
            'editing.contact_information.country.required' => __('Land is verplicht'),
        ];
    }

    public function mount()
    {
        $this->editing = $this->makeBlankEmployee();
    }

    public function resetFilters() {
        $this->useCachedRows();
        $this->reset('filters');
    }

    public function getModelRowsQueryProperty()
    {
        $query = Employee::with('contact_information')
            ->join('employee_contact_information as contact_information', 'contact_information.employee_id', '=', 'employees.id')
            ->search('name', $this->filters['search'])
            ->when($this->filters['active'], fn($query, $active) => $query->where('active', '=', $active));
        return $this->applySorting($query);
    }
    public function getModelRowsProperty()
    {
        return $this->cache(function () {
            return $this->applyPagination($this->modelrowsQuery);
        });
    }

    public function render()
    {
        if ($this->selectAll) {
           $this->selectPageRows();
        }
        return view('livewire.employees.employees-table', [
            'employees' => $this->modelrows,
        ]);
    }

    public function updatedFilters()
    {
        $this->resetPage();
    }

    public function exportSelected()
    {
        return response()->streamDownload(function () {
            echo $this->selectedRowsQuery->toCsv();
        }, 'Medewerkers.csv');

    }

    public function deleteSelected()
    {
        if ($this->SelectedRowsQuery->delete()) {
            $this->notify(trans('Medewerker(s) zijn verwijderd'));
        }
        $this->showDeleteModal = false;
    }

    public function makeBlankEmployee()
    {
        $employee = new Employee();
        $employee->load('contact_information');
        return $employee;
    }

    public function edit(Employee $employee)
    {
        $this->useCachedRows();
        $this->resetValidation();
        if ($this->editing->isNot($employee)) {
            $employee->load('contact_information');
            $this->editing = $employee;
        }
        $this->showEditModal = true;
    }

    public function create()
    {
        $this->useCachedRows();
        $this->resetValidation();
        if ($this->editing->getKey()) {
            $this->editing = $this->makeBlankEmployee();
            $this->editing->load('contact_information');
        }
        $this->showEditModal= true;
    }

    public function save()
    {
        $this->validate();
        // remove the relationship attributes. If not removed it will cause an error because it wants to save the field contact_information etc.
        $this->editing->setRawAttributes(
            Arr::except($this->editing->getAttributes(), 'contact_information')
        );
        if ($this->editing->save()) {
            $this->editing->contact_information->employee_id = $this->editing->id;
            if($this->editing->contact_information->save()) {
                $this->notify(trans('Medewerker is opgslagen'));
            }
        }

        $this->showEditModal = false;
    }

    public function toggleShowFilters()
    {
        $this->useCachedRows();
        $this->showFilters = ! $this->showFilters;
    }

    public function changeStatus(Employee $employee)
    {
        $employee->active = ! $employee->active;
        if($employee->save()) {
            $this->notify(trans('Status is veranderd'));
        } else {
            $this->notifyerror(trans('Er is iets mis gegaan'));
        }
    }
}
