<?php

namespace App\Http\Livewire\Companies;

use App\Traits\HasBulkActions;
use App\Traits\HasCachedRows;
use App\Traits\HasPerPagePagination;
use App\Traits\HasSort;
use Companies\Models\Company;
use Livewire\Component;

class CompaniesTable extends Component
{
    use HasPerPagePagination, HasSort, HasBulkActions, HasCachedRows;

    public $queryString = [];
    public $showEditModal = false;
    public $showFilters = false;
    public Company $editing;
    public $showDeleteModal = false;
    public $filters = [
        'search' => '',
        'active' => ''
    ];

    public function rules()
    {
        return [
            'editing.name' => 'required',
            'editing.email' => 'required|email',
            'editing.address' => 'nullable',
            'editing.number' => 'nullable|numeric',
            'editing.addon' => 'nullable',
            'editing.zip_code' => 'nullable',
            'editing.city' => 'nullable',
            'editing.country' => 'nullable',
            'editing.phone' => 'nullable|numeric',
            'editing.note' => 'nullable|max:1000',
            'editing.website' => 'nullable',
            'editing.kvk' => 'nullable|numeric',
            'editing.vat_number' => 'nullable',
        ];
    }

    public function messages()
    {
        return [
            'editing.name.required' => __('Naam is verplicht'),
            'editing.email.required' => __('Email is verplicht'),
            'editing.email.email' => __('Email is ongeldig'),
            'editing.number.numeric' => __('Straatnummer moet uit cijfers bestaan'),
            'editing.phone.numeric' => __('Telefoon moet uit cijfers bestaan'),
            'editing.note.max' => __('Notitie is te lang (max 1000)'),
            'editing.kvk.numeric' => __('KvK moet uit cijfers bestaan'),
        ];
    }

    public function mount()
    {
        $this->editing = $this->makeBlankCompany();
    }

    public function makeBlankCompany()
    {
        $company = new Company();
        return $company;
    }

    public function resetFilters()
    {
        $this->reset('filters');
    }

    public function getModelRowsQueryProperty()
    {
        $query = Company::query()
            ->search('name', $this->filters['search'])
            ->when($this->filters['active'], fn($query, $active) => $query->where('active', '=', $active));
        return $this->applySorting($query);
    }

    public function getModelRowsProperty()
    {
        return $this->cache(function () {
            return $this->applyPagination($this->modelrowsQuery);
        });
    }

    public function render()
    {
        if ($this->selectAll) {
            $this->selectPageRows();
        }
        return view('livewire.companies.companies-table', [
            'companies' => $this->modelrows,
        ]);
    }

    public function edit(Company $company)
    {
        $this->useCachedRows();
        $this->resetValidation();
        if ($this->editing->isNot($company)) {
            $this->editing = $company;
        }
        $this->showEditModal = true;
    }

    public function create()
    {
        $this->useCachedRows();
        $this->resetValidation();
        if ($this->editing->getKey()) {
            $this->editing = $this->makeBlankCompany();
        }
        $this->showEditModal = true;
    }

    public function save()
    {
        $this->validate();
        if ($this->editing->save()) {
            $this->notify(trans('Bedrijf is opgslagen'));
        }

        $this->showEditModal = false;
    }

    public function toggleShowFilters()
    {
        $this->useCachedRows();
        $this->showFilters = !$this->showFilters;
    }

    public function changeStatus(Company $company)
    {
        $company->active = !$company->active;
        if ($company->save()) {
            $this->notify(trans('Status is veranderd'));
        } else {
            $this->notifyerror(trans('Er is iets mis gegaan'));
        }
    }

    public function deleteSelected()
    {
        if ($this->SelectedRowsQuery->delete()) {
            $this->notify(trans('Bedrijven(s) zijn verwijderd'));
        }
        $this->showDeleteModal = false;
    }
}
